"""
Flask web server that runs scheduler in background thread.
Keeps Render.com free tier awake by responding to HTTP pings.
"""
from flask import Flask, jsonify
import threading
import schedule
import time
import os
import sys

app = Flask(__name__)
app.config['JSON_AS_ASCII'] = False
bot = None
ai_gen = None
scheduler_started = False

def init_bot():
    """Lazy initialization of bot"""
    global bot, ai_gen
    if bot is None:
        from bot import SocialMediaBot
        from ai_content_generator import AIContentGenerator
        bot = SocialMediaBot()
        ai_gen = AIContentGenerator()

def post_content(pillar, platform):
    """Generate and post content"""
    try:
        init_bot()
        if platform == 'twitter':
            content, _ = ai_gen.generate_post(pillar=pillar, platform='twitter')
            bot.post_to_twitter(content)
        elif platform == 'linkedin':
            content, _ = ai_gen.generate_post(pillar=pillar, platform='linkedin')
            bot.post_to_linkedin(content)
        elif platform == 'both':
            content, _ = ai_gen.generate_post(pillar=pillar, platform='both')
            bot.post_to_twitter(content['twitter'])
            bot.post_to_linkedin(content['linkedin'])
    except:
        pass

# Schedule all posts (24 per week)
schedule.every().monday.at("07:30").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().monday.at("12:00").do(lambda: post_content('learning_journey', 'twitter'))
schedule.every().monday.at("16:00").do(lambda: post_content('build_in_public', 'twitter'))
schedule.every().monday.at("19:00").do(lambda: post_content('learning_journey', 'linkedin'))

schedule.every().tuesday.at("08:00").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().tuesday.at("13:00").do(lambda: post_content('real_talk', 'twitter'))
schedule.every().tuesday.at("17:30").do(lambda: post_content('build_in_public', 'twitter'))

schedule.every().wednesday.at("07:00").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().wednesday.at("12:00").do(lambda: post_content('tech_threads', 'twitter'))
schedule.every().wednesday.at("16:30").do(lambda: post_content('learning_journey', 'twitter'))
schedule.every().wednesday.at("19:00").do(lambda: post_content('build_in_public', 'linkedin'))

schedule.every().thursday.at("07:30").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().thursday.at("11:30").do(lambda: post_content('learning_journey', 'twitter'))
schedule.every().thursday.at("15:00").do(lambda: post_content('real_talk', 'twitter'))
schedule.every().thursday.at("18:30").do(lambda: post_content('learning_journey', 'linkedin'))

schedule.every().friday.at("08:00").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().friday.at("12:30").do(lambda: post_content('build_in_public', 'twitter'))
schedule.every().friday.at("16:00").do(lambda: post_content('learning_journey', 'twitter'))

schedule.every().saturday.at("10:00").do(lambda: post_content('real_talk', 'twitter'))
schedule.every().saturday.at("14:00").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().saturday.at("18:00").do(lambda: post_content('build_in_public', 'linkedin'))

schedule.every().sunday.at("09:00").do(lambda: post_content('engagement_questions', 'twitter'))
schedule.every().sunday.at("13:00").do(lambda: post_content('learning_journey', 'twitter'))
schedule.every().sunday.at("17:00").do(lambda: post_content('real_talk', 'twitter'))

def run_scheduler():
    """Run scheduler in background thread"""
    while True:
        try:
            schedule.run_pending()
        except:
            pass
        time.sleep(60)

def start_scheduler():
    """Start scheduler on first real request"""
    global scheduler_started
    if not scheduler_started:
        scheduler_started = True
        scheduler_thread = threading.Thread(target=run_scheduler, daemon=True)
        scheduler_thread.start()

@app.route('/')
def home():
    """Health check endpoint"""
    start_scheduler()
    return "OK", 200, {'Content-Type': 'text/plain; charset=utf-8'}

@app.route('/health')
def health():
    """Health check for monitoring"""
    return "OK", 200, {'Content-Type': 'text/plain; charset=utf-8'}

@app.route('/status')
def status():
    """Get bot status"""
    return jsonify({"status": "running"})

if __name__ == '__main__':
    import os
    port = int(os.environ.get('PORT', 10000))
    app.run(host='0.0.0.0', port=port)
